%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Stability Regions of Numerical Methods for Solving Ordinary Differential
% Equations

% Created:     2025.12.02
% Last change: 2025.12.02
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
close all; clear; clc;
addpath(genpath('.'))
% Set latex as the intepreter for the text in the plot and default axes
% fontsize
set( 0, 'defaultAxesTickLabelInterpreter', 'latex' );
set( 0, 'defaultLegendInterpreter',        'latex' );
set( 0, 'defaultTextInterpreter',          'latex' );
set( 0, 'defaultAxesFontSize', 16 );   % or 14
%--------------------------------------------------------------------------
% Define some custom colors:
yellow = [0.894, 0.671, 0.094];
green = [0.667, 0.706, 0.118];
blue = [0.239, 0.376, 0.655];
red = [0.827, 0.341, 0.306];
orange = [0.870, 0.443, 0.137];
%--------------------------------------------------------------------------
% Data
% N = grid resolution
N = 200;

% Grid in complex plane
xrange = [-4 1];
yrange = [-4 4];
x = linspace(xrange(1), xrange(2), N);
y = linspace(yrange(1), yrange(2), N);
[X, Y] = meshgrid(x, y);
Z = X + 1i*Y;

%--------------------------------------------------------------------------
% Compute stability functions
% Explicit Euler
A = 0;
b = 1;
R_EE = computeRKStabilityRegion(A, b, Z);

% Heun
A = [0 0; 1 0];
b = [1/2; 1/2];
R_Heun = computeRKStabilityRegion(A, b, Z);

% Heun
A = [0 0;
     1 0];
b = [1/2;
     1/2];
R_Heun = computeRKStabilityRegion(A, b, Z);

% Classical RK4
A = [0   0   0 0;
     1/2 0   0 0;
     0   1/2 0 0;
     0   0   1 0];
b = [1/6; 1/3; 1/3; 1/6];
R_RK4 = computeRKStabilityRegion(A, b, Z);

%  Dormand–Prince RK45
A = [0               0               0               0               0               0               0;
     1/5             0               0               0               0               0               0;
     3/40            9/40            0               0               0               0               0;
     44/45          -56/15           32/9            0               0               0               0;
     19372/6561     -25360/2187      64448/6561     -212/729         0               0               0;
     9017/3168      -355/33          46732/5247      49/176         -5103/18656      0               0;
     35/384          0               500/1113        125/192        -2187/6784      11/84           0];

% Primary weights (i.e., those for the 5th order)
w1 = [35/384; 0; 500/1113; 125/192; -2187/6784; 11/84; 0];   % NOTE: must be 7 components

b = w1;

R_DP = computeRKStabilityRegion(A, b, Z);

%--------------------------------------------------------------------------
% Plot: tiled layout
figure('Position',[200 200 950 400]);
tl = tiledlayout(1,4);
title(tl, 'Stability Regions of Explicit Runge–Kutta Methods', 'FontSize', 16);

% Explicit Euler
nexttile;
contourf(X, Y, abs(R_EE), [0 1], 'LineStyle','none');
axis equal tight;
title('Explicit Euler');
xlabel('Re(z)');
ylabel('Im(z)');
grid on;

% Heun
nexttile;
contourf(X, Y, abs(R_Heun), [0 1], 'LineStyle','none');
axis equal tight;
title('Heun');
xlabel('Re(z)');
ylabel('Im(z)');
grid on;

% Classical RK4
nexttile;
contourf(X, Y, abs(R_RK4), [0 1], 'LineStyle','none');
axis equal tight;
title('RK4');
xlabel('Re(z)');
ylabel('Im(z)');
grid on;

% Dormand-Prince
nexttile;
contourf(X, Y, abs(R_DP), [0 1], 'LineStyle','none');
axis equal tight;
title('RK45 (Dormand-Prince)');
xlabel('Re(z)');
ylabel('Im(z)');
grid on;

% For changing the color of the regions:
colormap([red; 1 1 1]);