%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXERCISE
% Stiff ODEs.
% Numerical integration of the 1D heat equation.
% Created:     2025.12.05
% Last change: 2025.12.10
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
close all; clear; clc;
addpath(genpath('.'))
set( 0, 'defaultAxesTickLabelInterpreter', 'latex' );
set( 0, 'defaultLegendInterpreter',        'latex' );
set( 0, 'defaultTextInterpreter',          'latex' );
set( 0, 'defaultAxesFontSize', 14 );
%--------------------------------------------------------------------------
% Data
%--------------------------------------------------------------------------
Nx = 30;
Nt_vector = 1000*[1 2 4];

T = 2;

hx = 1/Nx;
x = hx*(1:Nx-1)';

u0 = x.^2.*(1-x);

L = diag(2*ones(Nx-1,1)) + diag(-ones(Nx-2,1),1) + diag(-ones(Nx-2,1),-1);
L = -L/hx^2;
%--------------------------------------------------------------------------

% Condition on h for stability of the explicit methods
% rhs_stability_condition = abs( 2/min(eig(A)) );
rhs_stability_condition = 0.5 * hx^2;

%--------------------------------------------------------------------------
blue = [0.239, 0.376, 0.655];
red = [0.827, 0.341, 0.306];
orange = [0.870, 0.443, 0.137];
%--------------------------------------------------------------------------
% Data
% N = grid resolution
N = 200;

% Grid in complex plane
xrange = [-4 1];
yrange = [-4 4];
x = linspace(xrange(1), xrange(2), N);
y = linspace(yrange(1), yrange(2), N);
[X, Y] = meshgrid(x, y);
Z = X + 1i*Y;

%--------------------------------------------------------------------------
% Compute stability functions
% Explicit Euler
A = 0;   % Runge-Kutta matrix
b = 1;   % weights
R_EE = computeRKStabilityRegion(A, b, Z);


M = numel(Nt_vector);
for j=1:M
    Nt = Nt_vector(j);
    ht = T/Nt;
    t = ht * (0:Nt)';

    if ht <= rhs_stability_condition
        is_satisfied = 'OK';
    else
        is_satisfied = 'NO :(';
    end
    fprintf( 'Current h: %.2e, \t Stability condition: %.2e, \t %s\n', ...
        ht, rhs_stability_condition, is_satisfied )

    u_EE = Explicit_Euler( u0, L, T, Nt );

    %----------------------------------------------------------------------
    figure(2*(j-1)+1)
    % Plotting the stability region
    contourf(X, Y, abs(R_EE), [0 1], 'LineStyle','none');
    axis equal tight;
    title('Explicit Euler');
    xlabel('Re(z)');
    ylabel('Im(z)');
    grid on;
    % For changing the color of the regions:
    colormap([blue; 1 1 1]);

    % % For drawing the unitary circle centered at (-1, 0)
    % theta = linspace(0, 2*pi, 400);   % Angle parameter
    % r = 1;                             % Radius
    % center = [-1, 0];                  % Center of the disk
    % 
    % x = center(1) + r*cos(theta);
    % y = center(2) + r*sin(theta);
    % 
    % plot(x, y, '-', 'Color', [0.65 0.65 0.65], 'LineWidth', 2);  % Gray dashed line
    hold on
    axis equal

    plot([-7.5, 1], [0, 0], '--', 'Color', 'k', 'LineWidth', 1.5);  % x axis
    plot([0, 0], [-2.5, 2.5], '--', 'Color', 'k', 'LineWidth', 1.5);  % y axis

    p2 = plot( ht*min(eig(L)), 0, 'ko', 'MarkerSize', 10, 'MarkerFaceColor', red );
    xlim([-7.5 1]);
    ylim([-2 2]);

    legend( p2, '$ h \cdot \lambda_{\mathrm{min}}(A) $' )
    %----------------------------------------------------------------------

    figure(2*j)
    colNorms = sqrt(sum(u_EE.^2,1)) * sqrt(hx);
    plot( t', colNorms, 'LineWidth', 2 )
    grid on
    xlabel('$ t $');
    ylabel('$ \| u(t) \| $');
    title('Norm of the Explicit Euler Solution');

end

pause

%--------------------------------------------------------------------------
% Data of the problem
% RHS
fun = @(t,u) L*u;


% Classical RK4
A = [0   0   0 0;
     1/2 0   0 0;
     0   1/2 0 0;
     0   0   1 0];
b = [1/6; 1/3; 1/3; 1/6];
R_RK4 = computeRKStabilityRegion(A, b, Z);

contourf(X, Y, abs(R_RK4), [0 1], 'LineStyle', 'none');
axis equal tight;
title('RK4');
xlabel('Re(z)');
ylabel('Im(z)');
grid on;
hold on;
% For changing the color of the regions:
blue = [0.239, 0.376, 0.655];
colormap([blue; 1 1 1]);

plot( ht*min(eig(L)), 0, 'ko', 'MarkerSize', 10, 'MarkerFaceColor', red );

pause


% Compute the numerical solution with the Dormand-Prince method with
% adaptive step-size control
[ t_DP, u_DP ] = RK45(fun, 0, T, u0, ht, 1e-6, L);

% colNorms = sqrt(sum(u_DP.^2,1)) * sqrt(hx);
%
% figure(8)
% plot( t_DP, colNorms, 'r', 'LineWidth', 2 )
% grid on
% xlabel('$ t $');
% ylabel('$ \| u(t) \| $');
% title('Norm of the Dormand-Prince Solution');
