function ynew = radau5_step( t, y, h, eps, options_newton )

% function ynew = radau5_step( t, y, h, eps, options_newton )
% Purpose: Computes one step of the Radau IIa method with an inner
%          simplified Newton method.
% Created:     2025.12.05
% Last change: 2025.12.07

[ A, b, c ] = radau5_coefficients();
s = 3;     % number of stages
d = length(y);

% Initial guess: all stages = y
Y = repmat(y,1,s);

J_small = vdp_jac( t, y, eps );
J_big = eye(s*d) - h * kron( A, J_small );

% The matrix J_big is the same for all iterations.
% Its LU-decomposition is done only once and is usually very costly. 
[ L, U, P ] = lu(J_big);

% Loop of the simplified Newton method
for it = 1:options_newton.maxiter
    % Initializations
    Z = zeros(d,s);

    % Compute RHS
    for i = 1:s
        Yi = Y(:,i);
        Z(:,i) = Yi - y;
        for j = 1:s
            Z(:,i) = Z(:,i) - h*A(i,j) * vdp_rhs(t + c(j)*h, Y(:,j), eps);
        end
    end

    % Solve the linear system using the LU decomposition:
    delta = -U\(L\(P*Z(:)));

    % Another possibility: use MATLAB's backslash directly
    % delta = -J_big \ Z(:);

    % We reshape it as a matrix:
    Y = Y + reshape(delta,d,s);

    if norm(delta,inf) < options_newton.tol
        break
    end
end

% Final update
ynew = y;
for i = 1:s
    ynew = ynew + h*b(i) * vdp_rhs(t + c(i)*h, Y(:,i), eps);
end

end
